program Turing_Machine;

uses Crt;

const
  TapeHeader = '// Chrob Software''s Turing Machine Tape File';
  ProgramHeader = '// Chrob Software''s Turing Machine Program Listing';
  ScrollClr = 14;
  BorderClr = 7;
  TitleClr = 15;
  TextClr = 11;
  InfoBarClr = 12;
  BlankClr = 4;
  DataClr = 10;
  HighlightClr = 13;

type
  ProgramPtr = ^ProgramType;
  ProgramType = record
                  StartState, StopState: Word;
                  StartCh, StopCh: Char;
                  Left: Boolean;
                  Next: ProgramPtr;
                end;
  TapePtr = ^TapeList;
  TapeList = record
               Data: Char;
               Prev, Next: TapePtr;
             end;
  TapeType = record
               Pos: TapePtr;
               State: Word;
             end;

var
  ProgramList: ProgramPtr;
  Tape: TapeType;
  Quiet, Pause, NoDelay, Aborted, OutState: Boolean;
  OutVal: Byte;

function StrToInt(St: String): LongInt;
var
  Int, ErrCode: Integer;
begin
  Val(St, Int, ErrCode);
  if ErrCode <> 0 then
    Int := 0;
  StrToInt := Int;
end;

function Upper(St: String): String;
var
  Loop: Byte;
begin
  for Loop := 1 to Length(St) do
    St[Loop] := UpCase(St[Loop]);
  Upper := St;
end;

{-=-=- Turns Cursor On -=-=-}
procedure CursorOn; assembler;
asm
  mov ah, 1
  mov ch, 7 {Top}
  mov cl, 8 {Bottom}
  int 10h
end;

{-=-=- Turns Cursor Off -=-=-}
procedure CursorOff; assembler;
asm
  mov ah, 1
  mov cl, 0
  int 10h
end;

{-=-=- Waits for user to press a key, then clears the Keyboard Buffer -=-=-}
procedure WaitKey;
begin
  ReadKey;
  while KeyPressed do ReadKey;
end;

{-=-=- Check to see if Specified file exists -=-=-}
function FileExists(FileName: String): Boolean;
var
  CheckFile: file;
begin
  {$I-}
  Assign(CheckFile, FileName);
  FileMode := 0;
  Reset(CheckFile);
  Close(CheckFile);
  {$I+}
  FileExists := (IOResult = 0) and (FileName <> '');
end;

{-=-=- Returns a string of NumChars lots of SChar -=-=-}
function MakeStr(SChar: Char; NumChars: Byte): String;
var
  TempStr: String;
begin
  for TempStr[0] := #1 to Chr(NumChars) do
    TempStr[Ord(TempStr[0])] := SChar;
  MakeStr := TempStr;
end;

function IntToStr(Num: LongInt): String;
var
  S: String[11];
begin
  Str(Num, S);
  IntToStr := S;
end;

{-=-=- Checks for valid Parameters -=-=-}
procedure CheckParams;
var
  ValidParams: Boolean;
  ErrorCode: Byte;
  Switches, ErrorStr: String;
begin
  ValidParams := True;
  ErrorCode := 0;
  if (ParamStr(1) = '') or (ParamStr(2) = '') or (ParamStr(3) = '')then
  begin
    ValidParams := False;
    ErrorCode := 1;
    ErrorStr := 'ERROR: Missing Parameter';
  end;
  if not(FileExists(ParamStr(1))) and (ErrorCode = 0) then
  begin
    ValidParams := False;
    ErrorCode := 2;
    ErrorStr := 'ERROR: Tape File does not exist';
  end;
  if not(FileExists(ParamStr(2))) and (ErrorCode = 0) then
  begin
    ValidParams := False;
    ErrorCode := 3;
    ErrorStr := 'ERROR: Program List File does not exist';
  end;

  if (ErrorCode = 0) then
  begin
    Switches := Upper(ParamStr(4));
    Quiet := False;
    Pause := False;
    NoDelay := False;
    OutState:= False;
    if Switches = '/QUIET' then
      Quiet := True
    else
      if Switches = '/PAUSE' then
        Pause := True
      else
        if Switches = '/NODELAY' then
          NoDelay := True
        else
          if (Switches = '/OUTSTATE:S') or
             (Switches = '/OUTSTATE:T') or
             (Switches = '/OUTSTATE:B') then
          begin
            OutState := True;
            OutVal := 0;
            case Switches[11] of
              'S': OutVal := 1;
              'T': OutVal := 2;
              'B': OutVal := 3;
            end;
            if OutVal = 0 then
            begin
              ErrorCode := 25;
              ErrorStr := 'ERROR: Invalid OutState Mode';
              ValidParams := False;
            end;
          end
          else
            if Switches <> '' then
            begin
              ValidParams := False;
              ErrorCode := 8;
              ErrorStr := 'ERROR: Invalid Parameter Specified';
            end;
  end;

  if (ErrorCode = 0) then
  begin
    Switches := Upper(ParamStr(5));
    if (Switches = '/OUTSTATE:S') or
       (Switches = '/OUTSTATE:T') or
       (Switches = '/OUTSTATE:B') then
    begin
      OutState := True;
      OutVal := 0;
      case Switches[11] of
        'S': OutVal := 1;
        'T': OutVal := 2;
        'B': OutVal := 3;
      end;
      if OutVal = 0 then
      begin
        ErrorCode := 25;
        ErrorStr := 'ERROR: Invalid OutState Mode';
        ValidParams := False;
      end;
    end
    else
      if Switches <> '' then
      begin
        ValidParams := False;
        ErrorCode := 9;
        ErrorStr := 'ERROR: Invalid Parameter Specified';
      end;
  end;

  if (Upper(ParamStr(4)) = Upper(ParamStr(5))) and (ParamStr(4) <> '') then
  begin
    ValidParams := False;
    ErrorCode := 9;
    ErrorStr := 'ERROR: Invalid Parameter Specified';
  end;

  if not(ValidParams) then
  begin
    WriteLn(ErrorStr);
    WriteLn('SYNTAX: TURING [Tape] [Program Listing] [Output Tape] <Switches> </OutState:?>');
    WriteLn;
    WriteLn('Switches are:');
    WriteLn('   /QUIET   - If specified, no output is displayed and calculations are');
    WriteLn('              done in the background. This can''t be used in conjunction');
    WriteLn('              with other switches');
    WriteLn('   /PAUSE   - Waits for a key to be pressed between each executed command.');
    WriteLn('              This can''t be used in conjunction with other switches');
    WriteLn('   /NODELAY - Doesn''t display the banner, so the program runs as fast as');
    WriteLn('              possible. This can''t be used in conjunction with other switches');
    WriteLn;
    WriteLn('/OUTSTATE:? - ? is S/T/B, where, if specified:');
    WriteLn('              S outputs States to file ''OUTSTATE.LST''');
    WriteLn('              T outputs Tapes to file ''OUTSTATE.TPE''');
    WriteLn('              B outputs Both to above files');
    WriteLn;
    WriteLn('NOTE: If [Output Tape] is specified as ''*'' then no output tape file is produced');
    Halt(ErrorCode);
  end;
end;

{-=-=- Initialises Tape and Program in Memory -=-=-}
procedure Init(var Tape: TapeType; var ProgramList: ProgramPtr);
begin
  Tape.Pos := nil;
  Tape.State := 0;
  ProgramList := nil;
end;

procedure ListError(Count: LongInt; FileLine: String; Position: Byte);
var
  Loop: Byte;
begin
  TextAttr := 7;
  WriteLn;
  Write('Error in Line ', Count, ': ');
  for Loop := 1 to Length(FileLine) do
  begin
    if Loop = Position then Write('<HERE>');
    Write(FileLine[Loop]);
  end;
  WriteLn;
  Halt(6);
end;

function ReadNum(St: String; var Counter: Byte; var Error: Boolean): Integer;
var
  Temp: Integer;
begin
  Error := False;
  if not(St[Counter] in ['0'..'9']) then
    Error := True;
  Temp := 0;
  while St[Counter] in ['0'..'9'] do
  begin
    Temp := (Temp * 10) + Ord(St[Counter]) - 48;
    Counter := Counter + 1;
  end;
  if St[Counter] in [',',' ',';'] then
    while St[Counter] in [',',' ',';'] do
      Counter := Counter + 1
  else
    Error := True;
  ReadNum := Temp;
end;

function ParseLine(FileLine: String; Count: LongInt): ProgramPtr;
var
  TempPtr: ProgramPtr;
  Loop, Loop2: Byte;
  Error: Boolean;
begin
  New(TempPtr);
  TempPtr^.Next := nil;
  for Loop2 := 1 to Length(FileLine) do
    if not(FileLine[Loop2] in [' ',';']) then
    begin
      Loop := Loop2;
      Loop2 := Length(FileLine);
    end;

  if FileLine[Loop] <> 'Q' then ListError(Count, FileLine, Loop);
  Inc(Loop);
  TempPtr^.StartState := ReadNum(FileLine, Loop, Error);
  if Error then ListError(Count, FileLine, Loop);
  TempPtr^.StartCh := UpCase(FileLine[Loop]);
  Inc(Loop);
  while FileLine[Loop] = ' ' do Inc(Loop);
  if FileLine[Loop] <> ',' then ListError(Count, FileLine, Loop);
  Inc(Loop);
  while FileLine[Loop] = ' ' do Inc(Loop);
  TempPtr^.StopCh := UpCase(FileLine[Loop]);
  Inc(Loop);
  while FileLine[Loop] = ' ' do Inc(Loop);
  if FileLine[Loop] <> ',' then ListError(Count, FileLine, Loop);
  Inc(Loop);
  while FileLine[Loop] = ' ' do Inc(Loop);
  case FileLine[Loop] of
    'L': TempPtr^.Left := True;
    'R': TempPtr^.Left := False;
  else
    ListError(Count, FileLine, Loop);
  end;
  Inc(Loop);
  while FileLine[Loop] = ' ' do Inc(Loop);
  if FileLine[Loop] <> ',' then ListError(Count, FileLine, Loop);
  Inc(Loop);
  while FileLine[Loop] = ' ' do Inc(Loop);
  if not(FileLine[Loop] in ['Q','H']) then ListError(Count, FileLine, Loop);
  if FileLine[Loop] = 'Q' then
  begin
    Inc(Loop);
    TempPtr^.StopState := ReadNum(FileLine, Loop, Error);
    if Error then ListError(Count, FileLine, Loop);
  end
  else
    TempPtr^.StopState := 0;
  ParseLine := TempPtr;
end;

procedure ReadLine(var LoadFile: Text; var FileLine: String);
begin
  ReadLn(LoadFile, FileLine);
  FileLine := Upper(FileLine);
  FileLine := FileLine + ';';
end;

procedure NextWait(var WaitCh: Char);
begin
  case WaitCh of
    '': WaitCh := '/';
    '/': WaitCh := '-';
    '-': WaitCh := '\';
    '\': WaitCh := '';
  end;
end;

function FirstCh(FileLine: String): Char;
var
  Loop: Byte;
begin
  for Loop := 1 to Length(FileLine) do
    if FileLine[Loop] <> ' ' then
    begin
      FirstCh := FileLine[Loop];
      Exit;
    end;
end;

{-=-=- Loads a Tape File into Tape List -=-=-}
function LoadTape(FileName: String; var Tape: TapeType): Boolean;
var
  LoadFile: Text;
  TempPtr: TapePtr;
  FileData: Char;
  Header: String;
  WaitCh: Char;
begin
  LoadTape := True;
  Assign(LoadFile, FileName);
  Reset(LoadFile);
  ReadLn(LoadFile, Header);
  if Header <> TapeHeader then
  begin
    Close(LoadFile);
    LoadTape := False;
    Exit;
  end;
  WaitCh := '';
  Write('Loading Tape File: ');
  ReadLn(LoadFile, Tape.State);
  New(TempPtr);
  TempPtr^.Next := nil;
  TempPtr^.Prev := nil;
  TempPtr^.Data := 'B';
  while not eof(LoadFile) do
  begin
    New(TempPtr^.Next);
    TempPtr^.Next^.Prev := TempPtr;
    TempPtr := TempPtr^.Next;
    Read(LoadFile, FileData);
    if FileData = '*' then
    begin
      Tape.Pos := TempPtr;
      Read(LoadFile, FileData);
    end;
    TempPtr^.Data := UpCase(FileData);
    TempPtr^.Next := nil;
    Write(WaitCh);
    GotoXY(WhereX - 1, WhereY);
    NextWait(WaitCh);
  end;
  Close(LoadFile);
  GotoXY(1, WhereY);
  WriteLn('Tape File Loaded Successfully');
end;

{-=-=- Loads a Program Text File into Program List -=-=-}
function LoadProgram(FileName: String; var ProgramList: ProgramPtr): Boolean;
var
  LoadFile: Text;
  FileLine: String;
  Count: LongInt;
  Loop: Byte;
  Tail: ProgramPtr;
  WaitCh: Char;
begin
  LoadProgram := True;
  Assign(LoadFile, FileName);
  Reset(LoadFile);
  ReadLn(LoadFile, FileLine);
  if FileLine <> ProgramHeader then
  begin
    Close(LoadFile);
    LoadProgram := False;
    Exit;
  end;
  Count := 1;
  FileLine := '';
  WaitCh := '';
  Write('Loading Program Listing: ');
  while (FileLine[0] = #0) or (FirstCh(FileLine) = ';') do
  begin
    ReadLine(LoadFile, FileLine);
    Write(WaitCh);
    GotoXY(WhereX - 1, WhereY);
    NextWait(WaitCh);
  end;
  ProgramList := ParseLine(FileLine, Count);
  Tail := ProgramList;
  while not eof(LoadFile) do
  begin
    ReadLine(LoadFile, FileLine);
    Inc(Count);
    if (FileLine[0] <> #0) and (FirstCh(FileLine) <> ';') then
    begin
      Tail^.Next := ParseLine(FileLine, Count);
      Tail := Tail^.Next;
      Write(WaitCh);
      GotoXY(WhereX - 1, WhereY);
      NextWait(WaitCh);
    end;
  end;
  Close(LoadFile);
  GotoXY(1, WhereY);
  WriteLn('Program Listing Loaded Successfully');
end;

{-=-=- Move Left on Tape once -=-=-}
procedure MoveLeft(var Pos: TapePtr);
begin
  if Pos^.Prev = nil then
  begin
    New(Pos^.Prev);
    Pos^.Prev^.Next := Pos;
    Pos := Pos^.Prev;
    Pos^.Prev := nil;
    Pos^.Data := 'B';
  end
  else
    Pos := Pos^.Prev;
end;

{-=-=- Moves Right on Tape once -=-=-}
procedure MoveRight(var Pos: TapePtr);
begin
  if Pos^.Next = nil then
  begin
    New(Pos^.Next);
    Pos^.Next^.Prev := Pos;
    Pos := Pos^.Next;
    Pos^.Next := nil;
    Pos^.Data := 'B';
  end
  else
    Pos := Pos^.Next;
end;

procedure ProgError(State: Word; Data: Char);
begin
  TextAttr := 7;
  GotoXY(1, 12);
  WriteLn('ERROR: No State found for Q', State, ', Data ', Data);
  Halt(7);
end;

function GetCommand(ProgramList: ProgramPtr; State: Word;
                    Data: Char): String;
var
  TempPtr: ProgramPtr;
  ExecLine: String;
begin
  TempPtr := ProgramList;
  while (TempPtr^.StartState <> State) or (TempPtr^.StartCh <> Data) do
  begin
    TempPtr := TempPtr^.Next;
    if TempPtr = nil then
    begin
      GetCommand := '';
      Exit;
    end;
  end;
  ExecLine := 'Q' + IntToStr(State) + ',' + Data + ',' + TempPtr^.StopCh;
  if TempPtr^.Left then
    ExecLine := ExecLine + ',L,'
  else
    ExecLine := ExecLine + ',R,';
  if TempPtr^.StopState = 0 then
    ExecLine := ExecLine + 'H'
  else
    ExecLine := ExecLine + 'Q' + IntToStr(TempPtr^.StopState);
  GetCommand := ExecLine;
end;

procedure FindCommand(ProgramList: ProgramPtr; var State: Word;
                      var Data: Char; var Left: Boolean);
var
  TempPtr: ProgramPtr;
begin
  TempPtr := ProgramList;
  while (TempPtr^.StartState <> State) or (TempPtr^.StartCh <> Data) do
  begin
    TempPtr := TempPtr^.Next;
    if TempPtr = nil then ProgError(State, Data);
  end;
  State := TempPtr^.StopState;
  Data := TempPtr^.StopCh;
  Left := TempPtr^.Left;
end;

procedure OutputTape(TapePos: TapePtr; FileName: String);
var
  TempPtr: TapePtr;
  SaveFile: Text;
begin
  TempPtr := TapePos;
  while TempPtr^.Prev <> nil do
    TempPtr := TempPtr^.Prev;
  Assign(SaveFile, FileName);
  ReWrite(SaveFile);
  WriteLn(SaveFile, TapeHeader);
  WriteLn(SaveFile, '0');
  while TempPtr <> nil do
  begin
    if TempPtr = TapePos then
      Write(SaveFile, '*');
    Write(SaveFile, TempPtr^.Data);
    TempPtr := TempPtr^.Next;
  end;
  Close(SaveFile);
end;

procedure WriteTape(TapePos: TapePtr; var SaveFile: Text);
var
  TempPtr: TapePtr;
begin
  TempPtr := TapePos;
  while TempPtr^.Prev <> nil do
    TempPtr := TempPtr^.Prev;
  while TempPtr <> nil do
  begin
    if TempPtr = TapePos then
      Write(SaveFile, '*');
    Write(SaveFile, TempPtr^.Data);
    TempPtr := TempPtr^.Next;
  end;
  WriteLn(SaveFile);
end;

{-=-=- Shifts a string 1 left and wraps characters -=-=-}
procedure ShiftStrL(var StrToShift: String);
var
  TempCh: Char;
  TempVal: Word;
begin
  if StrToShift[0] = #0 then Exit;
  TempCh := StrToShift[1];
  for TempVal := 1 to (Ord(StrToShift[0]) - 1) do
    StrToShift[TempVal] := StrToShift[TempVal + 1];
  StrToShift[Ord(StrToShift[0])] := TempCh;
end;

procedure UpdateDisplay(Tape: TapeType; ExecLine: String);
var
  TempPtr: TapePtr;
  Loop: Byte;
begin
  GotoXY(18, 4);
  WriteLn('      ');
  GotoXY(18, 4);
  TextAttr := InfoBarClr;
  WriteLn(Tape.State);
  GotoXY(60, 4);
  WriteLn('              ');
  GotoXY(60, 4);
  WriteLn(ExecLine);
  TempPtr := Tape.Pos;
  TextAttr := BlankClr;
  GotoXY(2, 6);
  Write('...');
  for Loop := 1 to 35 do
  begin
    GotoXY(40 - Loop, WhereY);
    if TempPtr^.Prev = nil then
    begin
      TextAttr := BlankClr;
      Write('B');
    end
    else
    begin
      TempPtr := TempPtr^.Prev;
      if TempPtr^.Data = 'B' then
        TextAttr := BlankClr
      else
        TextAttr := DataClr;
      Write(TempPtr^.Data);
    end;
  end;
  TempPtr := Tape.Pos;
  TextAttr := HighlightClr;
  GotoXY(40, 6);
  Write(TempPtr^.Data);
  for Loop := 1 to 36 do
  begin
    GotoXY(40 + Loop, WhereY);
    if TempPtr^.Next = nil then
    begin
      TextAttr := BlankClr;
      Write('B');
    end
    else
    begin
      TempPtr := TempPtr^.Next;
      if TempPtr^.Data = 'B' then
        TextAttr := BlankClr
      else
        TextAttr := DataClr;
      Write(TempPtr^.Data);
    end;
  end;
  TextAttr := BlankClr;
  Write('...');
  TextAttr := 7;
end;

procedure Scroll(var ScrollText: String);
var
  Loop: Byte;
begin
  if Pause then
  begin
    while not KeyPressed do
    begin
      ShiftStrL(ScrollText);
      GotoXY(2,8);
      TextAttr := ScrollClr;
      for Loop := 1 to 78 do
        Write(ScrollText[Loop]);
      Delay(70);
    end;
    while KeyPressed do ReadKey;
  end
  else
  begin
    if not(NoDelay) then
    begin
      ShiftStrL(ScrollText);
      GotoXY(2,8);
      TextAttr := ScrollClr;
      for Loop := 1 to 78 do
        Write(ScrollText[Loop]);
      Delay(70);
    end;
  end;
end;

{-=-=- Returns Key Value if One is Pressed -=-=-}
function IfKey: Integer;
var
  Key: Integer;
begin
  if KeyPressed then
  begin
    Key := Ord(ReadKey);
    if Key = 0 then
      IfKey := -Ord(ReadKey)
    else
      IfKey := Key
  end
  else
    IfKey := 0;
end;

procedure ExecuteTuring(var Tape: TapeType; ProgramList: ProgramPtr;
                        var Aborted: Boolean);
var
  Left: Boolean;
  Key: Integer;
  ScrollText: String;
  OutStateFile, OutTapeFile: Text;
begin
  if OutState then
  begin
    Assign(OutStateFile, 'OUTSTATE.LST');
    Assign(OutTapeFile, 'OUTSTATE.TPE');
    ReWrite(OutStateFile);
    ReWrite(OutTapeFile);
  end;
  Aborted := False;
  Key := 0;
  ScrollText := 'Turing Machine Emulator is Copyright(C) 1998 by Chrob Software Ltd. ';
  ScrollText := ScrollText + 'This program was written by Chris Fry and Robert Grant. 23/04/98. ';
  ScrollText := ScrollText + 'For further details, please E-Mail to: <fry@sucs.swan.ak> or ';
  ScrollText := ScrollText + '<148819.97@swansea.ac.uk>. Thankyou.                   ';
  if OutState then
    case OutVal of
      1: WriteLn(OutStateFile, Tape.State);
      2: WriteTape(Tape.Pos, OutTapeFile);
      3: begin
           WriteLn(OutStateFile, Tape.State);
           WriteTape(Tape.Pos, OutTapeFile);
         end;
    end;
  while (Tape.State <> 0) and (Key <> 27) do
  begin
    if not(Quiet) then
    begin
      UpdateDisplay(Tape, GetCommand(ProgramList, Tape.State, Tape.Pos^.Data));
      Scroll(ScrollText);
    end;
    TextAttr := 7;
    FindCommand(ProgramList, Tape.State, Tape.Pos^.Data, Left);
    if OutState then
      case OutVal of
        1: WriteLn(OutStateFile, Tape.State);
        2: WriteTape(Tape.Pos, OutTapeFile);
        3: begin
             WriteLn(OutStateFile, Tape.State);
             WriteTape(Tape.Pos, OutTapeFile);
           end;
      end;
    if Left then MoveLeft(Tape.Pos) else MoveRight(Tape.Pos);
    Key := IfKey;
  end;
  if not(Quiet) then
  begin
    UpdateDisplay(Tape, GetCommand(ProgramList, Tape.State, Tape.Pos^.Data));
    Scroll(ScrollText);
  end;
  if Key = 27 then Aborted := True;
  if OutState then
  begin
    Close(OutStateFile);
    Close(OutTapeFile);
  end;
end;

procedure MainScreen;
begin
  ClrScr;
  TextAttr := BorderClr;
  Write('', MakeStr('', 78), '');
  Write('', MakeStr(' ', 16));
  TextAttr := TitleClr;
  Write('T U R I N G    M A C H I N E    E M U L A T O R');
  TextAttr := BorderClr;
  Write(MakeStr(' ', 15), '');
  Write('', MakeStr('', 30), '', MakeStr('', 47), '');
  Write(' ');
  TextAttr := TextClr;
  Write('Current State: ');
  TextAttr := BorderClr;
  Write(MakeStr(' ', 14), ' ');
  TextAttr := TextClr;
  Write('Next Line to be Executed: ');
  TextAttr := BorderClr;
  Write(MakeStr(' ', 20), '');
  Write('', MakeStr('', 30), '', MakeStr('', 47), '');
  Write('', MakeStr(' ', 78), '');
  Write('', MakeStr('', 78), '');
  Write('', MakeStr(' ', 78), '');
  WriteLn('', MakeStr('', 78), '');
end;

begin
  TextAttr := 7;
  ClrScr;
  CursorOn;
  WriteLn('Turing Machine Emulator is Copyright(C) 1998 by Chrob Software Ltd.');
  WriteLn('This program was written by Chris Fry and Robert Grant. 23/04/98. ');
  WriteLn('For further details, please E-Mail to:');
  WriteLn('<fry@sucs.swan.ak> or <148819.97@swansea.ac.uk>.');
  WriteLn;
  CheckParams;
  Init(Tape, ProgramList);
  if not(LoadTape(ParamStr(1), Tape)) then
  begin
    WriteLn('ERROR: File is not a valid Chrob Software Tape Listing');
    Halt(4);
  end;
  if not(LoadProgram(ParamStr(2), ProgramList)) then
  begin
    WriteLn('ERROR: File is not a valid Chrob Software Turing Program File');
    Halt(5);
  end;
  if not(Quiet) then MainScreen;
  CursorOff;
  if Quiet then WriteLn('Please Wait - Compiling and Executing Turing Program ...');
  ExecuteTuring(Tape, ProgramList, Aborted);
  if Aborted then
  begin
    TextAttr := 7;
    if not(Quiet) then GotoXY(1, 11);
    WriteLn('Execution Aborted.');
    Halt(255);
  end;
  if Quiet then
  begin
    WriteLn('Complete');
  end
  else
    GotoXY(1, 11);
  CursorOn;
  if ParamStr(3) <> '*' then
  begin
    OutputTape(Tape.Pos, ParamStr(3));
    WriteLn('Tape Output File ''', Upper(ParamStr(3)), ''' has been Written');
  end
  else
    WriteLn('Tape Output Aborted.');
  if OutState then
    case OutVal of
      1: WriteLn('OutState State Written');
      2: WriteLn('OutState Tape Written');
      3: WriteLn('OutState Files Written');
    end;
  WriteLn;
end.
